<?php

namespace App\Services;

use App\Models\UniqueCode;
use Illuminate\Support\Facades\DB;

class UniqueCodeService
{
    private const MIN_CODE = 100;
    private const MAX_CODE = 999;
    private const EXPIRY_HOURS = 24;

    /**
     * Generate a unique 3-digit code (100-999) for a given base amount.
     * Uses database-level locking to prevent race conditions.
     *
     * @param float $baseAmount The subtotal + PPN before unique code
     * @return int The unique code (100-999)
     * @throws \Exception If no unique codes are available
     */
    public function generate(float $baseAmount): int
    {
        return DB::transaction(function () use ($baseAmount) {
            // First, expire any old reserved codes for this amount
            $this->expireOldCodes($baseAmount);

            // Try to find a recyclable code (available or expired)
            $recyclable = UniqueCode::where('base_amount', $baseAmount)
                ->whereIn('status', [UniqueCode::STATUS_AVAILABLE, UniqueCode::STATUS_EXPIRED])
                ->lockForUpdate()
                ->first();

            if ($recyclable) {
                $recyclable->update([
                    'status' => UniqueCode::STATUS_RESERVED,
                    'reserved_at' => now(),
                    'expires_at' => now()->addHours(self::EXPIRY_HOURS),
                    'order_id' => null,
                ]);
                return $recyclable->code;
            }

            // Get all codes currently in use for this amount
            $usedCodes = UniqueCode::where('base_amount', $baseAmount)
                ->whereIn('status', [UniqueCode::STATUS_RESERVED, UniqueCode::STATUS_USED])
                ->pluck('code')
                ->toArray();

            // Find available code
            $allCodes = range(self::MIN_CODE, self::MAX_CODE);
            $availableCodes = array_diff($allCodes, $usedCodes);

            if (empty($availableCodes)) {
                throw new \Exception('No unique codes available for this amount. Please try again later.');
            }

            // Pick a random available code
            $code = $availableCodes[array_rand($availableCodes)];

            // Create new unique code record
            UniqueCode::create([
                'code' => $code,
                'base_amount' => $baseAmount,
                'status' => UniqueCode::STATUS_RESERVED,
                'reserved_at' => now(),
                'expires_at' => now()->addHours(self::EXPIRY_HOURS),
            ]);

            return $code;
        });
    }

    /**
     * Mark a unique code as used when order is created.
     */
    public function markAsUsed(int $code, float $baseAmount, int $orderId): void
    {
        UniqueCode::where('code', $code)
            ->where('base_amount', $baseAmount)
            ->where('status', UniqueCode::STATUS_RESERVED)
            ->update([
                'status' => UniqueCode::STATUS_USED,
                'order_id' => $orderId,
            ]);
    }

    /**
     * Release a unique code (make it available again).
     * Called when an order is cancelled or payment completed.
     */
    public function release(int $code, float $baseAmount): void
    {
        UniqueCode::where('code', $code)
            ->where('base_amount', $baseAmount)
            ->update([
                'status' => UniqueCode::STATUS_AVAILABLE,
                'order_id' => null,
                'reserved_at' => null,
                'expires_at' => null,
            ]);
    }

    /**
     * Check if a code is available for reuse (payment already completed).
     */
    public function isCodeReusable(int $code, float $baseAmount): bool
    {
        $uniqueCode = UniqueCode::where('code', $code)
            ->where('base_amount', $baseAmount)
            ->first();

        if (!$uniqueCode) {
            return true;
        }

        // If previous order using this code is complete, it can be reused
        if ($uniqueCode->order && $uniqueCode->order->isComplete()) {
            return true;
        }

        // If code has expired
        if ($uniqueCode->hasExpired()) {
            return true;
        }

        return $uniqueCode->isAvailable();
    }

    /**
     * Expire old reserved codes for a given amount.
     */
    private function expireOldCodes(float $baseAmount): void
    {
        UniqueCode::where('base_amount', $baseAmount)
            ->where('status', UniqueCode::STATUS_RESERVED)
            ->whereNotNull('expires_at')
            ->where('expires_at', '<=', now())
            ->update(['status' => UniqueCode::STATUS_EXPIRED]);
    }

    /**
     * Clean up all expired codes (can be run via scheduler).
     */
    public function cleanupExpiredCodes(): int
    {
        return UniqueCode::where('status', UniqueCode::STATUS_RESERVED)
            ->whereNotNull('expires_at')
            ->where('expires_at', '<=', now())
            ->update(['status' => UniqueCode::STATUS_EXPIRED]);
    }
}
