<?php

namespace App\Services;

use App\Models\Order;
use Illuminate\Support\Facades\Log;

class PaymentVerificationService
{
    public function __construct(
        private MootaService $moota,
    ) {
    }

    /**
     * Process incoming Moota webhook and verify payments.
     *
     * @param array $payload Raw webhook payload
     * @return array{processed: int, matched: int, details: array}
     */
    public function processWebhook(array $payload): array
    {
        $mutations = $this->moota->parseWebhookPayload($payload);
        $processed = 0;
        $matched = 0;
        $details = [];

        foreach ($mutations as $mutation) {
            $processed++;

            // Only process credit (incoming) mutations
            if (($mutation['type'] ?? '') !== 'CR') {
                $details[] = [
                    'mutation_id' => $mutation['mutation_id'] ?? null,
                    'status' => 'skipped',
                    'reason' => 'Not a credit mutation',
                ];
                continue;
            }

            $amount = (float) $mutation['amount'];
            $order = $this->findPendingOrderByAmount($amount);

            if ($order) {
                $matched++;
                $this->confirmOrder($order, $mutation);
                $details[] = [
                    'mutation_id' => $mutation['mutation_id'] ?? null,
                    'amount' => $amount,
                    'order_id' => $order->id,
                    'order_number' => $order->order_number,
                    'status' => 'matched',
                ];

                Log::info('Payment matched via webhook', [
                    'order_id' => $order->id,
                    'order_number' => $order->order_number,
                    'mutation_id' => $mutation['mutation_id'] ?? null,
                    'amount' => $amount,
                ]);
            } else {
                $details[] = [
                    'mutation_id' => $mutation['mutation_id'] ?? null,
                    'amount' => $amount,
                    'status' => 'unmatched',
                ];
            }
        }

        Log::info('Moota webhook processed', [
            'total' => $processed,
            'matched' => $matched,
        ]);

        return [
            'processed' => $processed,
            'matched' => $matched,
            'details' => $details,
        ];
    }

    /**
     * Manual verification via Moota API polling (mutations).
     */
    public function verifyOrderPayment(Order $order, ?string $bankId = null): array
    {
        if (!$order->isPending()) {
            return [
                'verified' => false,
                'message' => 'Order is not pending',
                'mutation' => null,
            ];
        }

        $mutation = $this->moota->findMutationByAmount(
            amount: (float) $order->total_amount,
            bankId: $bankId ?? $order->bank->moota_bank_id ?? null,
            startDate: $order->created_at->format('Y-m-d'),
        );

        if ($mutation) {
            $this->confirmOrder($order, $mutation);

            return [
                'verified' => true,
                'message' => 'Payment verified and order confirmed',
                'mutation' => $mutation,
            ];
        }

        return [
            'verified' => false,
            'message' => 'Payment not found',
            'mutation' => null,
        ];
    }

    /**
     * Batch verify all pending orders via Moota API.
     *
     * @param string|null $bankId
     * @return array{total: int, verified: int, details: array}
     */
    public function verifyPendingOrders(?string $bankId = null): array
    {
        $pendingOrders = Order::pending()
            ->notExpired()
            ->orderBy('created_at', 'asc')
            ->limit(100) // Process in batches
            ->get();

        $total = $pendingOrders->count();
        $verified = 0;
        $details = [];

        foreach ($pendingOrders as $order) {
            $result = $this->verifyOrderPayment($order, $bankId);

            if ($result['verified']) {
                $verified++;
            }

            $details[] = [
                'order_id' => $order->id,
                'order_number' => $order->order_number,
                'amount' => $order->total_amount,
                'verified' => $result['verified'],
                'message' => $result['message'],
            ];
        }

        return [
            'total' => $total,
            'verified' => $verified,
            'details' => $details,
        ];
    }

    /**
     * Find pending order by exact total amount.
     */
    private function findPendingOrderByAmount(float $amount): ?Order
    {
        return Order::pending()
            ->notExpired()
            ->byAmount($amount)
            ->orderBy('created_at', 'asc')
            ->first();
    }

    /**
     * Confirm order with mutation data.
     */
    private function confirmOrder(Order $order, array $mutation): void
    {
        $mutationId = $mutation['mutation_id'] ?? $mutation['token'] ?? null;
        $order->markAsConfirmed($mutationId);
    }
}
