<?php

namespace App\Services;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

/**
 * Moota Service - Direct HTTP (no SDK)
 * 
 * Provides bank list and mutation list for payment verification.
 */
class MootaService
{
    private const BASE_URL = 'https://app.moota.co/api/v2';
    private const CACHE_BANK_TTL = 3600; // 1 hour

    private string $token;

    public function __construct()
    {
        $this->token = config('services.moota.api_token', '');

        if (empty($this->token)) {
            throw new \RuntimeException('MOOTA_API_TOKEN is not configured.');
        }
    }

    // =========================================================================
    // BANK ACCOUNTS
    // =========================================================================

    /**
     * Get list of registered bank accounts.
     */
    public function getBankAccounts(int $page = 1, int $perPage = 20): array
    {
        $cacheKey = "moota:banks:page_{$page}";

        if (Cache::has($cacheKey)) {
            return Cache::get($cacheKey);
        }

        $response = $this->request('GET', '/bank', [
            'page' => $page,
            'per_page' => $perPage,
        ]);

        Cache::put($cacheKey, $response, self::CACHE_BANK_TTL);

        return $response;
    }

    /**
     * Get all bank accounts (flat array).
     */
    public function getAllBankAccounts(): array
    {
        $result = $this->getBankAccounts(1, 100);

        return $result['data'] ?? [];
    }

    /**
     * Find a specific bank by ID.
     */
    public function findBank(string $bankId): ?array
    {
        return collect($this->getAllBankAccounts())
            ->firstWhere('bank_id', $bankId);
    }

    // =========================================================================
    // MUTATIONS
    // =========================================================================

    /**
     * Get mutations for a bank account.
     */
    public function getMutations(
        string $bankId,
        ?string $startDate = null,
        ?string $endDate = null,
        string $type = 'CR',
        int $page = 1
    ): array {
        $params = array_filter([
            'bank' => $bankId,
            'start_date' => $startDate,
            'end_date' => $endDate,
            'type' => $type,
            'page' => $page,
        ]);

        return $this->request('GET', '/mutation', $params);
    }

    /**
     * Find a matching CREDIT mutation by exact amount.
     */
    public function findMutationByAmount(
        float $amount,
        ?string $bankId = null,
        ?string $startDate = null
    ): ?array {
        $banks = $bankId
            ? [['bank_id' => $bankId]]
            : $this->getAllBankAccounts();

        $startDate = $startDate ?? now()->subDay()->format('Y-m-d');

        foreach ($banks as $bank) {
            $result = $this->getMutations(
                $bank['bank_id'],
                $startDate,
                now()->format('Y-m-d'),
                'CR'
            );

            $mutations = $result['data'] ?? [];

            foreach ($mutations as $mutation) {
                if (abs((float) $mutation['amount'] - $amount) < 0.01) {
                    return $mutation;
                }
            }
        }

        return null;
    }

    // =========================================================================
    // WEBHOOK
    // =========================================================================

    /**
     * Parse incoming Moota webhook payload.
     */
    public function parseWebhookPayload(array $payload): array
    {
        $mutations = [];

        foreach ($payload as $item) {
            if (!is_array($item)) {
                continue;
            }

            $mutations[] = [
                'mutation_id' => $item['mutation_id'] ?? $item['token'] ?? null,
                'bank_id' => $item['bank_id'] ?? null,
                'amount' => (float) ($item['amount'] ?? 0),
                'type' => $item['type'] ?? null,
                'description' => $item['description'] ?? '',
                'date' => $item['date'] ?? null,
            ];
        }

        return $mutations;
    }

    // =========================================================================
    // UTILITIES
    // =========================================================================

    /**
     * Clear Moota cache.
     */
    public function clearCache(): void
    {
        Cache::forget('moota:banks:page_1');
    }

    /**
     * Health check — ping the API.
     */
    public function healthCheck(): array
    {
        try {
            $result = $this->getBankAccounts(1, 1);

            return [
                'ok' => true,
                'bank_count' => $result['total'] ?? count($result['data'] ?? []),
            ];
        } catch (\Exception $e) {
            return ['ok' => false, 'error' => $e->getMessage()];
        }
    }

    // =========================================================================
    // HTTP
    // =========================================================================

    /**
     * Make an authenticated request to Moota API.
     */
    private function request(string $method, string $endpoint, array $params = []): array
    {
        try {
            $response = Http::withToken($this->token)
                        ->timeout(15)
                ->$method(self::BASE_URL . $endpoint, $params);

            if (!$response->successful()) {
                Log::error('Moota API error', [
                    'status' => $response->status(),
                    'body' => $response->body(),
                ]);
                throw new \RuntimeException("Moota API error: HTTP {$response->status()}");
            }

            return $response->json() ?? [];
        } catch (\Illuminate\Http\Client\ConnectionException $e) {
            Log::error('Moota API timeout', ['error' => $e->getMessage()]);
            throw new \RuntimeException('Moota API timeout: ' . $e->getMessage());
        }
    }
}
