<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class UniqueCode extends Model
{
    use HasFactory;

    public const STATUS_AVAILABLE = 'available';
    public const STATUS_RESERVED = 'reserved';
    public const STATUS_USED = 'used';
    public const STATUS_EXPIRED = 'expired';

    protected $fillable = [
        'code',
        'base_amount',
        'order_id',
        'status',
        'reserved_at',
        'expires_at',
    ];

    protected $casts = [
        'code' => 'integer',
        'base_amount' => 'decimal:2',
        'reserved_at' => 'datetime',
        'expires_at' => 'datetime',
    ];

    public function order(): BelongsTo
    {
        return $this->belongsTo(Order::class);
    }

    // Scopes
    public function scopeAvailable($query)
    {
        return $query->where('status', self::STATUS_AVAILABLE);
    }

    public function scopeReserved($query)
    {
        return $query->where('status', self::STATUS_RESERVED);
    }

    public function scopeExpired($query)
    {
        return $query->where('status', self::STATUS_EXPIRED)
            ->orWhere(function ($q) {
                $q->where('status', self::STATUS_RESERVED)
                    ->whereNotNull('expires_at')
                    ->where('expires_at', '<=', now());
            });
    }

    public function scopeForAmount($query, float $amount)
    {
        return $query->where('base_amount', $amount);
    }

    // Status helpers
    public function isAvailable(): bool
    {
        return $this->status === self::STATUS_AVAILABLE;
    }

    public function isReserved(): bool
    {
        return $this->status === self::STATUS_RESERVED;
    }

    public function isUsed(): bool
    {
        return $this->status === self::STATUS_USED;
    }

    public function hasExpired(): bool
    {
        if ($this->status === self::STATUS_EXPIRED) {
            return true;
        }

        return $this->isReserved() &&
            $this->expires_at &&
            $this->expires_at->isPast();
    }

    // Status transitions
    public function markAsUsed(int $orderId): void
    {
        $this->update([
            'order_id' => $orderId,
            'status' => self::STATUS_USED,
        ]);
    }

    public function markAsExpired(): void
    {
        $this->update([
            'status' => self::STATUS_EXPIRED,
            'order_id' => null,
        ]);
    }

    public function markAsAvailable(): void
    {
        $this->update([
            'status' => self::STATUS_AVAILABLE,
            'order_id' => null,
            'reserved_at' => null,
            'expires_at' => null,
        ]);
    }
}
