<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\DB;

class Product extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'description',
        'stock',
        'price',
        'is_active',
    ];

    protected $casts = [
        'stock' => 'integer',
        'price' => 'decimal:2',
        'is_active' => 'boolean',
    ];

    public function orderItems(): HasMany
    {
        return $this->hasMany(OrderItem::class);
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeInStock($query)
    {
        return $query->where('stock', '>', 0);
    }

    /**
     * Atomically decrement stock using database-level operation.
     * Returns true if successful, false if insufficient stock.
     */
    public function decrementStock(int $quantity): bool
    {
        $affected = DB::table('products')
            ->where('id', $this->id)
            ->where('stock', '>=', $quantity)
            ->update(['stock' => DB::raw("stock - {$quantity}")]);

        if ($affected > 0) {
            $this->refresh();
            return true;
        }

        return false;
    }

    /**
     * Atomically increment stock.
     */
    public function incrementStock(int $quantity): void
    {
        DB::table('products')
            ->where('id', $this->id)
            ->update(['stock' => DB::raw("stock + {$quantity}")]);

        $this->refresh();
    }
}
