<?php

namespace App\Http\Controllers;

use App\Models\Bank;
use App\Models\Product;
use App\Models\StoreLocation;
use App\Services\OrderService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class BookingController extends Controller
{
    public function __construct(
        private OrderService $orderService,
    ) {
    }

    /**
     * Show the booking form.
     */
    public function index()
    {
        $products = Product::active()->inStock()->get();
        $locations = StoreLocation::active()->get();
        $banks = Bank::active()->get();

        // Get default bank for payment details
        $defaultBank = $banks->first();

        return view('booking.index', compact('products', 'locations', 'banks', 'defaultBank'));
    }

    /**
     * Process booking submission.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'customer_name' => 'required|string|max:255',
            'customer_whatsapp' => 'required|string|max:20',
            'customer_address' => 'required|string|max:1000',
            'notes' => 'nullable|string|max:500',
            'store_location_id' => 'required|exists:store_locations,id',
            'bank_id' => 'required|exists:banks,id',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
        ], [
            'customer_name.required' => 'Nama lengkap wajib diisi.',
            'customer_whatsapp.required' => 'Nomor WhatsApp wajib diisi.',
            'customer_address.required' => 'Alamat wajib diisi.',
            'store_location_id.required' => 'Pilih lokasi store.',
            'items.required' => 'Pilih minimal satu produk.',
            'items.min' => 'Pilih minimal satu produk.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $order = $this->orderService->createOrder(
                customerData: [
                    'name' => $request->customer_name,
                    'whatsapp' => $request->customer_whatsapp,
                    'address' => $request->customer_address,
                    'notes' => $request->notes,
                ],
                items: $request->items,
                storeLocationId: $request->store_location_id,
                bankId: $request->bank_id,
            );

            return response()->json([
                'success' => true,
                'message' => 'Pesanan berhasil dibuat!',
                'order' => [
                    'id' => $order->id,
                    'order_number' => $order->order_number,
                    'subtotal' => $order->subtotal,
                    'ppn' => $order->ppn,
                    'unique_code' => $order->unique_code,
                    'total_amount' => $order->total_amount,
                    'expired_at' => $order->expired_at->format('d M Y H:i'),
                    'bank' => [
                        'name' => $order->bank->bank_name,
                        'account_name' => $order->bank->account_name,
                        'account_number' => $order->bank->account_number,
                    ],
                    'store_location' => $order->storeLocation->name,
                ],
                'redirect_url' => route('booking.confirmation', $order->order_number),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 400);
        }
    }

    /**
     * Show order confirmation page.
     */
    public function confirmation(string $orderNumber)
    {
        $order = \App\Models\Order::where('order_number', $orderNumber)
            ->with(['items', 'storeLocation', 'bank'])
            ->firstOrFail();

        return view('booking.confirmation', compact('order'));
    }

    /**
     * Check order status (AJAX).
     */
    public function checkStatus(string $orderNumber)
    {
        $order = \App\Models\Order::where('order_number', $orderNumber)->first();

        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Order tidak ditemukan.',
            ], 404);
        }

        return response()->json([
            'success' => true,
            'status' => $order->status,
            'is_paid' => $order->isConfirmed() || $order->isComplete(),
            'is_expired' => $order->isExpired(),
        ]);
    }

    /**
     * Upload payment proof.
     */
    public function uploadPaymentProof(Request $request, string $orderNumber)
    {
        $order = \App\Models\Order::where('order_number', $orderNumber)->firstOrFail();

        $request->validate([
            'payment_proof' => 'required|image|mimes:jpeg,jpg,png,webp|max:5120', // 5MB max
        ], [
            'payment_proof.required' => 'File bukti transfer wajib diupload.',
            'payment_proof.image' => 'File harus berupa gambar.',
            'payment_proof.mimes' => 'Format file harus JPEG, PNG, atau WebP.',
            'payment_proof.max' => 'Ukuran file maksimal 5MB.',
        ]);

        try {
            // Delete old file if exists
            if ($order->payment_proof && file_exists(public_path($order->payment_proof))) {
                unlink(public_path($order->payment_proof));
            }

            // Store new file
            $file = $request->file('payment_proof');
            $filename = $order->order_number . '_' . time() . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('uploads'), $filename);

            $order->update([
                'payment_proof' => 'uploads/' . $filename,
            ]);

            return back()->with('success', 'Bukti transfer berhasil diupload. Kami akan segera memverifikasi pembayaran Anda.');
        } catch (\Exception $e) {
            return back()->with('error', 'Gagal mengupload file: ' . $e->getMessage());
        }
    }
}
