<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\StoreLocation;
use Illuminate\Http\Request;

class StoreLocationController extends Controller
{
    public function index(Request $request)
    {
        $query = StoreLocation::query();

        if ($request->filled('search')) {
            $query->where('name', 'like', "%{$request->search}%");
        }

        if ($request->filled('status')) {
            $query->where('is_active', $request->status === 'active');
        }

        $locations = $query->latest()->paginate(15);

        return view('admin.locations.index', compact('locations'));
    }

    public function create()
    {
        return view('admin.locations.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'gmap_link' => 'nullable|url|max:500',
            'is_active' => 'boolean',
        ]);

        $validated['is_active'] = $request->boolean('is_active', true);

        StoreLocation::create($validated);

        return redirect()->route('admin.locations.index')
            ->with('success', 'Lokasi store berhasil ditambahkan.');
    }

    public function edit(StoreLocation $location)
    {
        return view('admin.locations.edit', compact('location'));
    }

    public function update(Request $request, StoreLocation $location)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'gmap_link' => 'nullable|url|max:500',
            'is_active' => 'boolean',
        ]);

        $validated['is_active'] = $request->boolean('is_active', true);

        $location->update($validated);

        return redirect()->route('admin.locations.index')
            ->with('success', 'Lokasi store berhasil diperbarui.');
    }

    public function destroy(StoreLocation $location)
    {
        // Check if location has orders
        if ($location->orders()->exists()) {
            return back()->with('error', 'Tidak dapat menghapus lokasi yang memiliki order.');
        }

        $location->delete();

        return redirect()->route('admin.locations.index')
            ->with('success', 'Lokasi store berhasil dihapus.');
    }
}
