<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\Product;
use App\Models\StoreLocation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ReportController extends Controller
{
    public function index(Request $request)
    {
        $dateFrom = $request->input('date_from', now()->startOfMonth()->format('Y-m-d'));
        $dateTo = $request->input('date_to', now()->format('Y-m-d'));
        $status = $request->input('status');
        $storeLocationId = $request->input('store_location_id');

        $query = Order::query();

        if ($status) {
            $query->where('status', $status);
        } else {
            // Default: only paid orders
            $query->whereIn('status', ['confirm', 'complete']);
        }

        if ($storeLocationId) {
            $query->where('store_location_id', $storeLocationId);
        }

        $query->whereDate('created_at', '>=', $dateFrom)
            ->whereDate('created_at', '<=', $dateTo);

        // Summary statistics
        $summary = [
            'total_orders' => (clone $query)->count(),
            'total_revenue' => (clone $query)->sum('total_amount'),
            'total_subtotal' => (clone $query)->sum('subtotal'),
            'total_ppn' => (clone $query)->sum('ppn'),
            'average_order' => (clone $query)->avg('total_amount') ?? 0,
        ];

        // Revenue by day
        $revenueByDay = (clone $query)
            ->select(
                DB::raw('DATE(created_at) as date'),
                DB::raw('SUM(total_amount) as total'),
                DB::raw('COUNT(*) as count')
            )
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Revenue by store location
        $revenueByLocation = (clone $query)
            ->select(
                'store_location_id',
                DB::raw('SUM(total_amount) as total'),
                DB::raw('COUNT(*) as count')
            )
            ->groupBy('store_location_id')
            ->with('storeLocation')
            ->get();

        // Top products
        $topProducts = DB::table('order_items')
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->whereIn('orders.status', ['confirm', 'complete'])
            ->whereDate('orders.created_at', '>=', $dateFrom)
            ->whereDate('orders.created_at', '<=', $dateTo)
            ->when($storeLocationId, fn($q) => $q->where('orders.store_location_id', $storeLocationId))
            ->select(
                'order_items.product_name',
                DB::raw('SUM(order_items.quantity) as total_qty'),
                DB::raw('SUM(order_items.subtotal) as total_revenue')
            )
            ->groupBy('order_items.product_name')
            ->orderByDesc('total_qty')
            ->limit(10)
            ->get();

        // Status breakdown
        $statusBreakdown = Order::whereDate('created_at', '>=', $dateFrom)
            ->whereDate('created_at', '<=', $dateTo)
            ->when($storeLocationId, fn($q) => $q->where('store_location_id', $storeLocationId))
            ->select('status', DB::raw('COUNT(*) as count'))
            ->groupBy('status')
            ->get()
            ->pluck('count', 'status');

        $storeLocations = StoreLocation::all();

        return view('admin.reports.index', compact(
            'summary',
            'revenueByDay',
            'revenueByLocation',
            'topProducts',
            'statusBreakdown',
            'storeLocations',
            'dateFrom',
            'dateTo',
            'status',
            'storeLocationId',
        ));
    }
}
